using System;
using System.Windows.Forms;
using ComponentPro.Net.Mail;

namespace ScanMessagesSample
{
    public partial class Signature : Form
    {
        private static int _uniqueCode = 1000; // Uniquely incremental bounce signature identifier.
        private readonly BounceInspector _filter; // The BounceInspector object.
        private bool _modifyMode; // Modify?
        private int _categoryCode, _typeCode; // Bounce Category and Bounce Type infomation.
        private string _pattern; // The Regex pattern for this signature.
        private bool _delete; // Indicates whether hard bounce will be deleted.
        private BounceSignaturePart _part; // Indicates which part of the processing message will be analyzed.

        /// <summary>
        /// Initializes a new instance of the class.
        /// </summary>
        /// <param name="filter">The BounceInspector object.</param>
        /// <param name="modifyMode">A boolean flags indicating the modifying mode.</param>
        public Signature(BounceInspector filter, bool modifyMode)
        {
            InitializeComponent();

            _filter = filter;

            // Load up the bounce categories and fill them into the Category combo box.
            foreach (BounceCategory bc in _filter.BounceCategories.Values)
            {
                cbxCategory.Items.Add(bc.Name);
                if (bc.Code == _categoryCode)
                    cbxCategory.Text = bc.Name;
            }

            // Load up the bounce types and fill them into the Type combo box.
            foreach (BounceType bt in _filter.BounceTypes.Values)
            {
                cbxType.Items.Add(bt.Name);
                if (bt.Code == _typeCode)
                    cbxType.Text = bt.Name;
            }

            _modifyMode = modifyMode;            
        }

        /// <summary>
        /// Handles the Form Load event.
        /// </summary>
        /// <param name="e">The event arguments.</param>
        protected override void OnLoad(EventArgs e)
        {
            base.OnLoad(e);

            // Modify signature?
            if (_modifyMode)
            {
                // Change the title
                this.Text = "Modify custom signature";
                // Set text.
                txtPattern.Text = _pattern;
                
                chkContent.Checked = (_part & BounceSignaturePart.Body) == BounceSignaturePart.Body;
                chkSubject.Checked = (_part & BounceSignaturePart.Subject) == BounceSignaturePart.Subject;
                chkBodyHtml.Checked = (_part & BounceSignaturePart.BodyHtml) == BounceSignaturePart.BodyHtml;
                chkHeaders.Checked = (_part & BounceSignaturePart.Headers) == BounceSignaturePart.Headers;
                chkAttachmentsName.Checked = (_part & BounceSignaturePart.AttachmentName) == BounceSignaturePart.AttachmentName;
                chkAttachmentsContent.Checked = (_part & BounceSignaturePart.AttachmentContent) == BounceSignaturePart.AttachmentContent;

                chkDelete.Checked = _delete;
                chkAnalyzeFirst.Enabled = false;
                cbxCategory.Text = _filter.BounceCategories[_categoryCode].Name;
                cbxType.Text = _filter.BounceTypes[_typeCode].Name;
            }
            else
                this.Text = "Add a new custom signature";
        }

        public bool ModifyMode
        {
            get { return _modifyMode; }
            set { _modifyMode = value; }
        }

        public int CategoryCode
        {
            get { return _categoryCode; }
            set { _categoryCode = value; }
        }

        public int TypeCode
        {
            get { return _typeCode; }
            set { _typeCode = value; }
        }

        public string Pattern
        {
            get { return _pattern; }
            set { _pattern = value; }
        }

        public bool Delete
        {
            get { return _delete; }
            set { _delete = value; }
        }

        public BounceSignaturePart Part
        {
            get { return _part; }
            set { _part = value; }
        }

        public bool CheckFirst
        {
            get { return chkAnalyzeFirst.Checked; }
        }

        /// <summary>
        /// Handles the OK button's Click event.
        /// </summary>
        /// <param name="sender">The button object.</param>
        /// <param name="e">The event arguments.</param>
        private void btnOK_Click(object sender, EventArgs e)
        {
            DialogResult = DialogResult.None;

            // Validate the form controls.
            if (txtPattern.Text.Length == 0)
            {
                MessageBox.Show("Please enter signature regex pattern", "BounceInspector", MessageBoxButtons.OK);                
                return;
            }

            if (cbxCategory.Text.Length == 0)
            {
                MessageBox.Show("Please choose a category", "BounceInspector", MessageBoxButtons.OK);
                return;
            }

            if (cbxType.Text.Length == 0)
            {
                MessageBox.Show("Please choose a type", "BounceInspector", MessageBoxButtons.OK);
                return;
            }

            bool found = false;

            // Check to see whether the category already exists.
            foreach (BounceCategory bc in _filter.BounceCategories.Values)
                if (string.Compare(bc.Name, cbxCategory.Text, true) == 0)
                {
                    found = true;
                    _categoryCode = bc.Code;
                    break;
                }
            if (!found)
            {
                // If the category does not exist, then add it to the Bounce Category list.
                _filter.AddBounceCategory(++_uniqueCode, cbxCategory.Text);
                _categoryCode = _uniqueCode;
            }

            found = false;
            // Check to see whether the type already exists.
            foreach (BounceType bt in _filter.BounceTypes.Values)
                if (string.Compare(bt.Name, cbxType.Text, true) == 0)
                {
                    found = true;
                    _typeCode = bt.Code;
                    break;
                }
            if (!found)
            {
                // If the category does not exist, then add it to the Bounce Type list.
                _filter.AddBounceType(++_uniqueCode, cbxType.Text);
                _typeCode = _uniqueCode;
            }

            _part = BounceSignaturePart.None;
            if (chkContent.Checked) // If user wants to analyze body text.
                _part |= BounceSignaturePart.Body;
            if (chkSubject.Checked) // If user wants to analyze subject.
                _part |= BounceSignaturePart.Subject;
            if (chkBodyHtml.Checked) // If user wants to analyze body html.
                _part |= BounceSignaturePart.BodyHtml;
            if (chkHeaders.Checked) // If user wants to analyze message headers.
                _part |= BounceSignaturePart.Headers;
            if (chkAttachmentsName.Checked) // If user wants to analyze attachments' name.
                _part |= BounceSignaturePart.AttachmentName;
            if (chkAttachmentsContent.Checked) // If user wants to analyze attachments' content.
                _part |= BounceSignaturePart.AttachmentContent;

            if (_part == BounceSignaturePart.None) // No part selected
            {
                MessageBox.Show("Please choose at least one part (body text, html, subject,...)", "BounceInspector", MessageBoxButtons.OK);
                return;
            }

            _delete = chkDelete.Checked;

            _pattern = txtPattern.Text;

            this.DialogResult = DialogResult.OK;
        }
    }
}