using System;
using System.Collections.Generic;
using System.Windows.Forms;
using ComponentPro.Net.Mail;

namespace ScanMessagesSample
{
    public partial class CustomSignatures : Form
    {
        private readonly List<SignatureInfo> _signatures; // A local list containing information of signatures.
        private readonly BounceInspector _filter; // BounceInspector object.

        public CustomSignatures()
        {
        }

        public CustomSignatures(BounceInspector filter, List<SignatureInfo> signatures)
        {
            InitializeComponent();
            _filter = filter;
            _signatures = signatures;

            // Init the list control with the given signatures.
            foreach (SignatureInfo si in _signatures)
            {
                ListViewItem item = new ListViewItem(new string[] {si.RegexPattern, _filter.BounceCategories[si.CategoryCode].Name, 
                                        _filter.BounceTypes[si.TypeCode].Name, si.Delete.ToString(), si.SignaturePart.ToString()});


                ltvCustomSignatures.Items.Add(item);
            }
        }

        /// <summary>
        /// Handles the Add button's Click event.
        /// </summary>
        /// <param name="sender">The button object.</param>
        /// <param name="e">The event arguments.</param>
        private void btnAdd_Click(object sender, EventArgs e)
        {
            // Create a new Signature information class.
            Signature dlg = new Signature(_filter, false);
            DialogResult r = dlg.ShowDialog();
            
            if (r == DialogResult.OK)
            {
                SignatureInfo si = new SignatureInfo();
                si.CategoryCode = dlg.CategoryCode;
                si.Delete = dlg.Delete;
                si.RegexPattern = dlg.Pattern;
                si.SignaturePart = dlg.Part;
                si.TypeCode = dlg.TypeCode;
                // Add a new bounce signature to the BounceInspector object.
                si.Index = _filter.AddBounceSignature(si.RegexPattern, si.CategoryCode, si.TypeCode, si.Delete, dlg.CheckFirst);

                // Also add it's info to the list control.
                ListViewItem item = new ListViewItem(new string[] {si.RegexPattern, _filter.BounceCategories[si.CategoryCode].Name, 
                                        _filter.BounceTypes[si.TypeCode].Name, si.Delete.ToString(), si.SignaturePart.ToString()});


                // If this signature is marked as check first (highest position in the bouncefilter object's signature list).
                if (dlg.CheckFirst)
                {
                    _signatures.Insert(0, si);
                    ltvCustomSignatures.Items.Insert(0, item);
                }
                else
                {
                    _signatures.Add(si);
                    ltvCustomSignatures.Items.Add(item);
                }
            }
        }

        /// <summary>
        /// Handles the Modify button's Click event.
        /// </summary>
        /// <param name="sender">The button object.</param>
        /// <param name="e">The event arguments.</param>
        private void btnModify_Click(object sender, EventArgs e)
        {
            // Show error when no item is selected.
            if (ltvCustomSignatures.SelectedItems.Count == 0)
            {
                MessageBox.Show("No signature selected", "BounceInspector", MessageBoxButtons.OK);
                return;
            }

            int index = ltvCustomSignatures.SelectedItems[0].Index;
            // Get signature info.
            SignatureInfo si = _signatures[index];

            Signature dlg = new Signature(_filter, true);
            dlg.CategoryCode = si.CategoryCode;
            dlg.TypeCode = si.TypeCode;
            dlg.Pattern = si.RegexPattern;
            dlg.Delete = si.Delete;
            dlg.Part = si.SignaturePart;

            if (dlg.ShowDialog() == DialogResult.OK)
            {                
                si.CategoryCode = dlg.CategoryCode;
                si.Delete = dlg.Delete;
                si.RegexPattern = dlg.Pattern;
                si.SignaturePart = dlg.Part;
                si.TypeCode = dlg.TypeCode;

                // Update signature information (you can use this method to modify built-in signatures).
                _filter.ModifyBounceSignature(si.Index, si.RegexPattern, si.CategoryCode, si.TypeCode, si.Delete, si.SignaturePart);

                // Update the selected list item.
                ltvCustomSignatures.SelectedItems[0].SubItems[0].Text = si.RegexPattern;
                ltvCustomSignatures.SelectedItems[0].SubItems[1].Text = _filter.BounceCategories[si.CategoryCode].Name;
                ltvCustomSignatures.SelectedItems[0].SubItems[2].Text = _filter.BounceTypes[si.TypeCode].Name;
                ltvCustomSignatures.SelectedItems[0].SubItems[3].Text = si.Delete.ToString();
                ltvCustomSignatures.SelectedItems[0].SubItems[4].Text = si.SignaturePart.ToString();
            }
        }

        /// <summary>
        /// Handles the Remove button's Click event.
        /// </summary>
        /// <param name="sender">The button object.</param>
        /// <param name="e">The event arguments.</param>
        private void btnRemove_Click(object sender, EventArgs e)
        {
            // Show error when no item is selected.
            if (ltvCustomSignatures.SelectedItems.Count == 0)
            {
                MessageBox.Show("No signature selected", "BounceInspector", MessageBoxButtons.OK);
                return;
            }

            int index = ltvCustomSignatures.SelectedItems[0].Index;

            // Get signature info.
            SignatureInfo s = _signatures[index];

            // Remove the signature from the filter object.
            _filter.RemoveSignature(s.Index);

            // Remove the signature information from the local list as well.
            _signatures.RemoveAt(index);

            // Update other signatures' index
            for (int i = index; i < _signatures.Count; i++)
            {
                _signatures[i].Index--;
            }

            // Remove the selected list item.
            ltvCustomSignatures.Items.RemoveAt(index);
        }
    }
}